from flask import Flask, render_template, request, redirect, url_for, jsonify
import mysql.connector
from mysql.connector import Error
import os
from datetime import datetime

app = Flask(__name__)

# Database configuration - UPDATE THESE WITH YOUR CREDENTIALS
DB_CONFIG = {
    'host': 'localhost',
    'database': 'osdb_recover',
    'user': 'root',  # Change this to your MySQL user
    'password': 'your_password_here'  # Change this to your MySQL password
}

def get_db_connection():
    """Create and return a database connection"""
    try:
        connection = mysql.connector.connect(**DB_CONFIG)
        return connection
    except Error as e:
        print(f"Error connecting to MySQL: {e}")
        return None

def get_client_ip():
    """Get the client's IP address"""
    if request.headers.get('X-Forwarded-For'):
        return request.headers.get('X-Forwarded-For').split(',')[0]
    return request.remote_addr

@app.route('/')
def index():
    """Home page showing recent approved quotes"""
    connection = get_db_connection()
    if not connection:
        return "Database connection error", 500
    
    cursor = connection.cursor(dictionary=True)
    cursor.execute("""
        SELECT id, quote, comment, rox, sox, sux, score 
        FROM quotes 
        WHERE approved = '1' 
        ORDER BY id DESC 
        LIMIT 25
    """)
    quotes = cursor.fetchall()
    cursor.close()
    connection.close()
    
    return render_template('index.html', quotes=quotes, page_title='Recent Quotes')

@app.route('/browse')
@app.route('/browse/<int:page>')
def browse(page=1):
    """Browse all quotes with pagination"""
    per_page = 25
    offset = (page - 1) * per_page
    
    connection = get_db_connection()
    if not connection:
        return "Database connection error", 500
    
    cursor = connection.cursor(dictionary=True)
    
    # Get total count
    cursor.execute("SELECT COUNT(*) as total FROM quotes WHERE approved = '1'")
    total = cursor.fetchone()['total']
    total_pages = (total + per_page - 1) // per_page
    
    # Get quotes for current page
    cursor.execute("""
        SELECT id, quote, comment, rox, sox, sux, score 
        FROM quotes 
        WHERE approved = '1' 
        ORDER BY id DESC 
        LIMIT %s OFFSET %s
    """, (per_page, offset))
    quotes = cursor.fetchall()
    cursor.close()
    connection.close()
    
    return render_template('browse.html', quotes=quotes, page=page, 
                         total_pages=total_pages, page_title='Browse Quotes')

@app.route('/top')
@app.route('/top/<int:page>')
def top(page=1):
    """Show top-rated quotes"""
    per_page = 25
    offset = (page - 1) * per_page
    
    connection = get_db_connection()
    if not connection:
        return "Database connection error", 500
    
    cursor = connection.cursor(dictionary=True)
    
    # Get total count
    cursor.execute("SELECT COUNT(*) as total FROM quotes WHERE approved = '1'")
    total = cursor.fetchone()['total']
    total_pages = (total + per_page - 1) // per_page
    
    # Get top quotes
    cursor.execute("""
        SELECT id, quote, comment, rox, sox, sux, score 
        FROM quotes 
        WHERE approved = '1' 
        ORDER BY score DESC, rox DESC 
        LIMIT %s OFFSET %s
    """, (per_page, offset))
    quotes = cursor.fetchall()
    cursor.close()
    connection.close()
    
    return render_template('top.html', quotes=quotes, page=page, 
                         total_pages=total_pages, page_title='Top Quotes')

@app.route('/quote/<int:quote_id>')
def view_quote(quote_id):
    """View a single quote"""
    connection = get_db_connection()
    if not connection:
        return "Database connection error", 500
    
    cursor = connection.cursor(dictionary=True)
    cursor.execute("""
        SELECT id, quote, comment, rox, sox, sux, score 
        FROM quotes 
        WHERE id = %s AND approved = '1'
    """, (quote_id,))
    quote = cursor.fetchone()
    cursor.close()
    connection.close()
    
    if not quote:
        return "Quote not found", 404
    
    return render_template('quote.html', quote=quote, page_title=f'Quote #{quote_id}')

@app.route('/vote/<int:quote_id>/<vote_type>', methods=['POST'])
def vote(quote_id, vote_type):
    """Handle voting on quotes"""
    if vote_type not in ['rox', 'sox', 'sux']:
        return jsonify({'error': 'Invalid vote type'}), 400
    
    client_ip = get_client_ip()
    connection = get_db_connection()
    if not connection:
        return jsonify({'error': 'Database connection error'}), 500
    
    cursor = connection.cursor(dictionary=True)
    
    # Check if user has already voted on this quote
    cursor.execute("SELECT * FROM votes WHERE qid = %s AND ip = %s", (quote_id, client_ip))
    existing_vote = cursor.fetchone()
    
    if existing_vote:
        cursor.close()
        connection.close()
        return jsonify({'error': 'You have already voted on this quote'}), 403
    
    # Record the vote
    cursor.execute("INSERT INTO votes (qid, ip) VALUES (%s, %s)", (quote_id, client_ip))
    
    # Update the quote's vote count
    cursor.execute(f"UPDATE quotes SET {vote_type} = {vote_type} + 1 WHERE id = %s", (quote_id,))
    
    # Update the score (rox - sux)
    cursor.execute("UPDATE quotes SET score = rox - sux WHERE id = %s", (quote_id,))
    
    # Get updated vote counts
    cursor.execute("SELECT rox, sox, sux, score FROM quotes WHERE id = %s", (quote_id,))
    updated_quote = cursor.fetchone()
    
    connection.commit()
    cursor.close()
    connection.close()
    
    return jsonify({
        'success': True,
        'rox': updated_quote['rox'],
        'sox': updated_quote['sox'],
        'sux': updated_quote['sux'],
        'score': updated_quote['score']
    })

@app.route('/submit', methods=['GET', 'POST'])
def submit():
    """Submit a new quote"""
    if request.method == 'POST':
        quote_text = request.form.get('quote', '').strip()
        comment = request.form.get('comment', '').strip()
        
        if not quote_text:
            return render_template('submit.html', error='Quote cannot be empty', page_title='Submit Quote')
        
        client_ip = get_client_ip()
        connection = get_db_connection()
        if not connection:
            return render_template('submit.html', error='Database connection error', page_title='Submit Quote')
        
        cursor = connection.cursor()
        
        # Get the next ID
        cursor.execute("SELECT MAX(id) as max_id FROM quotes")
        result = cursor.fetchone()
        next_id = (result[0] or 0) + 1
        
        # Insert the quote (approved = '0' for pending moderation)
        cursor.execute("""
            INSERT INTO quotes (id, quote, comment, ip, rox, sox, sux, score, approved)
            VALUES (%s, %s, %s, %s, 0, 0, 0, 0, '0')
        """, (next_id, quote_text, comment, client_ip))
        
        connection.commit()
        cursor.close()
        connection.close()
        
        return render_template('submit.html', success=True, page_title='Submit Quote')
    
    return render_template('submit.html', page_title='Submit Quote')

@app.route('/random')
def random_quote():
    """Redirect to a random quote"""
    connection = get_db_connection()
    if not connection:
        return "Database connection error", 500
    
    cursor = connection.cursor(dictionary=True)
    cursor.execute("""
        SELECT id FROM quotes 
        WHERE approved = '1' 
        ORDER BY RAND() 
        LIMIT 1
    """)
    quote = cursor.fetchone()
    cursor.close()
    connection.close()
    
    if quote:
        return redirect(url_for('view_quote', quote_id=quote['id']))
    return redirect(url_for('index'))

if __name__ == '__main__':
    app.run(debug=True, host='0.0.0.0', port=5000)
